(* TEST.PAS ****************************************************************)
(* Program przykladowy pokazujacy sposob wykorzystania drivera modulu      *)
(* LC-011-0812 firmy AMBEX.                                                *)
(***************************************************************************)
(***************************************************************************)
(* UWAGA: Przed rozpoczeciem kompilacji nalezy sprawdzic ustawienie opcji  *)
(*	kompilatora:                                                       *)
(*	      - alingment = byte,                                          *)
(***************************************************************************)

program test;

uses crt, dos;

{$i ambex-lc.pas}

type
    buftype = array[1..320] of integer;		(* bufor na probki         *)

var
    s_init:      lc0_init;
    s_total:     lc0_total;
    s_module:    lc0_module;
    s_info:      lc0_info;
    s_break:     lc0_break;
    s_analog_in: lc0_analog_in;
    s_leave:     lc0_leave;

    r: registers;			(* parametr dla funkcji intr       *)
    LCinterrupt: integer;		(* numer przerwania driver'a       *)
    modulenum: integer;			(* numer badanego modulu           *)

const
    BUFLEN	= 320;				(* dlugosc bufora          *)
    SAMPLES	= 20;				(* liczba probek           *)
    CHANNELS	= 8;				(* liczba kanalow          *)

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Zamiana cyfry szesnastkowej na jej reprezentacje znakowa.             *)
(* Parametry:                                                              *)
(*   dig - cyfra do zamiany                                                *)
(* Wartosc:                                                                *)
(*   Reprezentacja znakowa cyfry.                                          *)
(***************************************************************************)
function hexdigit(dig: word) : char;
begin
    if dig < 10 then hexdigit := chr(ord('0') + dig)
    else             hexdigit := chr(ord('A') + dig - 10);
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wydrukowanie liczby calkowitej w postaci szesnastkowej.               *)
(* Parametry:                                                              *)
(*   val - wartosc do wydrukowania                                         *)
(***************************************************************************)
procedure writehex(val: word);
begin
    write(hexdigit((val and $F000) shr 12));
    write(hexdigit((val and $F00) shr 8));
    write(hexdigit((val and $F0) shr 4));
    write(hexdigit(val and $F));
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Inicjalizacja zmiennych.                                              *)
(***************************************************************************)
procedure initprogram;
begin
    s_init.LC0_CODE		:= MODULE_INIT;
    s_total.LC0_CODE		:= GET_TOTAL_CONFIGURATION;
    s_module.LC0_CODE		:= GET_MODULE_CONFIGURATION;
    s_info.LC0_CODE		:= GET_INFO;
    s_break.LC0_CODE		:= BREAK;
    s_analog_in.LC0_CODE	:= ANALOG_INPUT;
    s_leave.LC0_CODE		:= LEAVE_DRIVER;
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Sprawdzenie obecnosci drivera.                                        *)
(* Sposob:                                                                 *)
(*   Przez probe otwarcia urzadzenia o nazwie okreslonej przez driver.     *)
(* Parametry:                                                              *)
(*   name - nazwa urzadzenia                                               *)
(* Wartosc:                                                                *)
(*   TRUE  - driver jest zainstalowany                                     *)
(*   FALSE - driver nie jest zainstalowany                                 *)
(***************************************************************************)
function driverinstalled(name: string) : boolean;
var
    hd: text;
begin
{$I-}
    assign(hd, name);
    reset(hd);
    if IOResult <> 0 then
	driverinstalled := false
    else
	begin
	    close(hd);
	    driverinstalled := true
	end;
{$I+}
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Rozpoznanie czy driver jest zainstalowany.                            *)
(* Uwagi:                                                                  *)
(*   Funkcja dodatkowo wyswietla komunikat o testowanym module.            *)
(*   Jezeli driver nie jest zainstalowany to po wyswietleniu wlasciwego    *)
(*   komunikatu program konczy prace.                                      *)
(***************************************************************************)
procedure checkdrivers;
begin
    clrscr;
    gotoxy(1, 1);
    if driverinstalled('LC1108^^') then
	begin
	    writeln('Test modulu LC-011-0812');
	    LCinterrupt := LC011_08;
	end
    else
	begin
	    writeln('Driver nie jest zainstalowany!');
	    halt;
	end;
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Rozpoznanie konfiguracji badanego modulu.                             *)
(* Sposob:                                                                 *)
(*   Przez wykorzystanie funkcji driver'a GET_TOTAL_CONFIGURATION,         *)
(*   GET_MODULE_CONFIGURATION, GET_INFO.                                   *)
(* Uwagi:                                                                  *)
(*   Funkcja nadaje wartosc zmiennej modulenum (numer badanego modulu)     *)
(*   wypelnia struktury total, module, info i inicjalizuje zainstalowane   *)
(*    moduly.                                                              *)
(***************************************************************************)
procedure askdriver;
label	lab1;
begin
    r.dx := seg(s_total);
    r.di := ofs(s_total);
    intr(LCinterrupt, r);			(* GET_TOTAL_CONFIGURATION *)

				(* inicjalizacja zainstalowanych modulow   *)
    s_init.LC0_IMODULE := s_total.LC0_TONF and $F;
    r.dx := seg(s_init);
    r.di := ofs(s_init);
    intr(LCinterrupt, r);				(* MODULE_INIT     *)

					(* sprawdzenie, ktory modul jest   *)
					(* zainstalowany: A, B, C czy D    *)
    for modulenum := 1 to 4 do
	if ((s_total.LC0_TONF and (1 shl (modulenum - 1))) <> 0) then
	    goto lab1;

lab1:
					(* spytanie o konfiguracje modulu  *)
    s_module.LC0_MMODULE := modulenum;
    r.dx := seg(s_module);
    r.di := ofs(s_module);
    intr(LCinterrupt, r);			(* GET_MODULE_CONFIGURATION*)

					(* spytanie o konfiguracje toru a/c*)
    s_info.LC0_GMODULE := modulenum;
    s_info.LC0_GTYPE   := LC0_AINPUT;
    s_info.LC0_GNUM    := 1;
    r.dx := seg(s_info);
    r.di := ofs(s_info);
    intr(LCinterrupt, r);				(* GET_INFO        *)
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wyswietlenie konfiguracji badanego modulu i jego toru a/c.            *)
(* Sposob:                                                                 *)
(*   Przez wykorzystanie informacji zawartych w strukturach module i info. *)
(***************************************************************************)
procedure displayconfiguration;
var
    i: integer;
begin
					(* konfiguracja modulu              *)
    writeln;
    writeln('---------- Konfiguracja modulu:');
    writeln;
    write('Adres modulu: ');
    writehex(s_module.LC0_MBASE1);
    writeln(' (hex)');
    writeln('Liczba przetwornikow a/c: ', s_module.LC0_MIAD);
    writeln('Liczba przetwornikow c/a: ', s_module.LC0_MIDA);
    writeln('Liczba portow cyfrowych wejsciowych: ', s_module.LC0_MIDI);
    writeln('Liczba portow cyfrowych wyjsciowych: ', s_module.LC0_MIDO);
    writeln('Czestotliwosc zegara modulu:  ',
	    (s_module.LC0_MCLOCK / 1000):4:2, 'MHz');

					(* konfiguracja toru a/c            *)
    writeln;
    writeln('---------- Konfiguracja toru a/c:');
    writeln;
    writeln('Liczba kanalow: ', s_info.LC0_GCHAN);
    writeln('Rozdzielczosc: ', s_info.LC0_GRES, ' bitow');
    writeln('Zakres napiec: ', (s_info.LC0_GMINV / 10):5:1, '..',
	    (s_info.LC0_GMAXV / 10):4:1, ' V');
    if s_info.LC0_GDMA = $FF then
	writeln('Tor a/c nie jest podlaczony do kanalow DMA')
    else
	writeln('Numer kanalu DMA podlaczonego do toru a/c: ',
							s_info.LC0_GDMA);
    writeln('Minimalne okresy probkowania [s]:');
    write('     ');
    for i := 1 to s_info.LC0_GCHAN do
	begin
	    write((s_info.LC0_GMINP[i] / 10):5:1);
	    if i < s_info.LC0_GCHAN then
		begin
		    write(', ');
		    if (i mod 8) = 0 then
			begin
			    writeln;	(* zlamanie linii co 8 wielkosci   *)
			    write('     ');
			end
		end
	end;
    writeln;
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Zainstalowanie procedury obslugi przerwania generowanego przez        *)
(*   Ctrl-Break.                                                           *)
(* Sposob:                                                                 *)
(*   Przez wywolanie funkcji BREAK.                                        *)
(***************************************************************************)
procedure installbreak;
begin
    s_break.LC0_BMODE := LC0_BREAK_INST;
    s_break.LC0_BPROC := nil;	(* podlozona zostanie procedura driver'a   *)
    r.dx := seg(s_break);
    r.di := ofs(s_break);
    intr(LCinterrupt, r);				(* BREAK           *)
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Przerwa miedzy kolejnymi czesciami programu - oczekiwanie na reakcje  *)
(*   operatora.                                                            *)
(***************************************************************************)
procedure pressanykey;
var
    c: char;
begin
    writeln;
    writeln('Nacisnij dowolny klawisz . . .');
    c := readkey;
    clrscr;
    gotoxy(1, 1);
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wyswietlenie bufora z danymi pomiarowymi.                             *)
(* Parametry:                                                              *)
(*   buf - adres bufora                                                    *)
(*   c   - liczba kanalow                                                  *)
(*   s   - liczba probek na kanal                                          *)
(***************************************************************************)
procedure displaybuf(buf: buftype; c, s: integer);
var
    i: integer;
    j: integer;
begin
    for i := 1 to s do
	begin
	    write(i:2, ': ');
	    for j := 1 to c do
		begin
		    writehex(buf[(i - 1) * c + j]);
		    write('  ');
		end;
	    writeln;
	end;
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wypisanie komunikatow o bledzie / ostrzezeniu / dodatkowej informacji *)
(*   bledzie.                                                              *)
(* Parametry:                                                              *)
(*   status   - LC0_STATUS                                                 *)
(*   err_stat - LC0_ERR_STAT                                               *)
(***************************************************************************)
procedure drivererror(status, err_stat: shortint);
begin
    if status > 0 then
	writeln('--------- Ostrzezenie: ', status)
    else
	writeln('--------- Blad: ', status);
    if err_stat < 0 then
	writeln('--------- Informacje dodatkowe: ', err_stat);
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wykonanie transmisji blokowej.                                        *)
(* Sposob:                                                                 *)
(*   Przez wykonanie funkcji ANALOG_INPUT.                                 *)
(* Parametry:                                                              *)
(*   start - typ warunku startu                                            *)
(* Wartosc:                                                                *)
(* Uwagi:                                                                  *)
(*   Funkcja wywolujaca musi ustawic okres probkowania i parametry warunku *)
(*   startu.                                                               *)
(***************************************************************************)
procedure transmission(start: byte);
var
    buf: buftype;			(* bufor na probki                 *)
    c: char;
begin
    writeln('Pomiar blokowy, ', CHANNELS, ' kanalow, ', SAMPLES,
	    ' probek, okres probkowania ',
	    (s_analog_in.LC0_APER / 10):11:1, ' s:');
    case start of
	LC0_SIMMED:
	    writeln('Warunek startu: natychmiast');
	LC0_STIME:
	    writeln('Warunek startu: uplyw ', s_analog_in.LC0_ASTART.time,
		    ' sekund');
				(* pozostale warunki nie sa obslugiwane bo *)
				(* w programie nie sa wykorzystywane       *)
	end;
    s_analog_in.LC0_AMODULE := modulenum;
    s_analog_in.LC0_ANUM := 1;
    s_analog_in.LC0_AMODE := LC0_MOD_START or
			     LC0_MOD_NEW_PAR or
			     LC0_MOD_SYNCHR or
			     LC0_MOD_BLOCK;
				(* stop po zmierzeniu okreslonej liczby    *)
				(* probek                                  *)
    s_analog_in.LC0_ASTST := start + LC0_ZSAMPLES;
				(* praca wielokanalowa, CHANNELS kanalow   *)
    s_analog_in.LC0_ACHAN := CHANNELS;
    s_analog_in.LC0_AADDR := @buf;
    s_analog_in.LC0_ALEN := BUFLEN;	(* dlugosc bufora                  *)
    s_analog_in.LC0_ABMAR := 0;
    s_analog_in.LC0_AEMAR := 0;		(* oba marginesy zerowe            *)
					(* calkowita liczba probek         *)
    s_analog_in.LC0_ASTOP.samples := SAMPLES * CHANNELS;
    r.dx := seg(s_analog_in);
    r.di := ofs(s_analog_in);
    intr(LCinterrupt, r);				(* ANALOG_INPUT    *)
    if s_analog_in.LC0_STATUS <> LC0_OK then
	begin
	    if s_analog_in.LC0_STATUS = LC0_BROKEN then
	    	c := readkey;	(* oproznienie buf. klawiatury po Ctrl-Break*)
	    drivererror(s_analog_in.LC0_STATUS, s_analog_in.LC0_ERR_STAT)
	end
    else
	displaybuf(buf, CHANNELS, SAMPLES);
    pressanykey;
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wykonanie poprawnej transmisji blokowej i wyswietlenie zmierzonych    *)
(*   wartosci.                                                             *)
(* Sposob:                                                                 *)
(*   Przez wykonanie funkcji transmission.                                 *)
(***************************************************************************)
procedure blocktransmission;
begin
    writeln('---------- Poprawne przetwarzanie blokowe');
    writeln;
    s_analog_in.LC0_APER := s_info.LC0_GMINP[CHANNELS];
    transmission(LC0_SIMMED);
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wykonanie blednej transmisji blokowej.                                *)
(* Sposob:                                                                 *)
(*   Przez wykonanie funkcji ANALOG_INPUT z okresem probkowania mniejszym  *)
(*   niz minimalny wskazany przez driver.                                  *)
(***************************************************************************)
procedure failblocktransmission;
begin
    writeln('---------- Bledne przetwarzanie blokowe');
    writeln;
    s_analog_in.LC0_APER := s_info.LC0_GMINP[CHANNELS] - 1;
    transmission(LC0_SIMMED);
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wykonanie poprawnej transmisji blokowej ale przerwanej przez operatora*)
(*   (Ctrl-Break) w trakcie czekania na spelnienie warunku startu.         *)
(* Sposob:                                                                 *)
(*   Przez wykonanie funkcji ANALOG_INPUT z warunkiem startu LC0_STIME     *)
(*   (start po okreslonym czasie) i parametrem tego warunku - 1000s.       *)
(***************************************************************************)
procedure interruptedbefore;
begin
    writeln('---------- Przetwarzanie blokowe z oczekiwaniem 1000s');
    writeln('---------- Oczekiwanie nalezy przerwac Ctrl-Break');
    writeln;
    s_analog_in.LC0_APER := s_info.LC0_GMINP[CHANNELS];
    s_analog_in.LC0_ASTART.time := 1000;
    transmission(LC0_STIME);
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Wykonanie pomiaru bloku probek za pomoca tranmsisji pojedynczej, przy *)
(*   czym caly blok ma byc zmierzony po 5s od startu.                      *)
(* Sposob:                                                                 *)
(*   Przez wykonanie funkcji ANALOG_INPUT.                                 *)
(* Parametry:                                                              *)
(* Wartosc:                                                                *)
(* Uwagi:                                                                  *)
(***************************************************************************)
procedure singletransmission;
var
    buf: buftype;			(* bufor na probki                  *)
    i: integer;
    j: integer;
begin
    writeln('---------- Przetwarzanie pojedyncze - po 5 sekundach');
    writeln;
    s_analog_in.LC0_AMODULE := modulenum;
    s_analog_in.LC0_ANUM := 1;
    s_analog_in.LC0_AMODE := LC0_MOD_START or
			     LC0_MOD_NEW_PAR or
			     LC0_MOD_SINGLE;
				(* praca wielokanalowa, CHANNELS kanalow   *)
    s_analog_in.LC0_ACHAN := CHANNELS;
    s_analog_in.LC0_AADDR := @buf;
    s_analog_in.LC0_ALEN := CHANNELS;	(* dlugosc bufora (tylko na jeden  *)
    					(* pomiar)                         *)
    s_analog_in.LC0_ASTST := LC0_STIME;
    s_analog_in.LC0_ASTART.time := 5;	(* start po 5 sekundach            *)
    r.dx := seg(s_analog_in);
    r.di := ofs(s_analog_in);
    intr(LCinterrupt, r);				(* ANALOG_INPUT    *)
    s_analog_in.LC0_ASTST := LC0_SIMMED;
    s_analog_in.LC0_AADDR := @buf[CHANNELS + 1];
    i := 3;
    r.dx := seg(s_analog_in);
    r.di := ofs(s_analog_in);
    intr(LCinterrupt, r);				(* ANALOG_INPUT    *)
					(* zgaszenie bitu LC0_MOD_NEW_PAR  *)
    s_analog_in.LC0_AMODE := s_analog_in.LC0_AMODE and (not LC0_MOD_NEW_PAR);
    for j := 1 to 18 do
	begin
	    s_analog_in.LC0_AADDR := @buf[(i - 1) * CHANNELS + 1];
	    i := i + 1;
	    r.dx := seg(s_analog_in);
	    r.di := ofs(s_analog_in);
	    intr(LCinterrupt, r);			(* ANALOG_INPUT    *)
	end;
    displaybuf(buf, CHANNELS, SAMPLES);
    pressanykey;
end;

(***************************************************************************)
(* Przeznaczenie:                                                          *)
(*   Zakonczenie pracy programu.                                           *)
(* Sposob:                                                                 *)
(*   Rozstanie sie z driver'em za pomoca funkcji LEAVE_DRIVER.             *)
(***************************************************************************)
procedure quit;
begin
    r.dx := seg(s_leave);
    r.di := ofs(s_leave);
    intr(LCinterrupt, r);				(* LEAVE_DRIVER    *)

    writeln('Dziekuje, to wszystko!');
end;


begin
	initprogram;		(* inicjalizacja programu                  *)
	checkdrivers;		(* rozpoznanie zainstalowanego driver'a    *)
	askdriver;		(* odpytanie driver'a o konfiguracje       *)
	displayconfiguration;	(* wyswietlenie konfiguracji modulu i      *)
				(* toru a/c                                *)
	installbreak;		(* zainstalowanie procedury obslugi        *)
				(* przerwania generowanego przez Ctrl-Break*)
	pressanykey;		(* czekanie na operatora                   *)
	blocktransmission;	(* wykonanie transmisji blokowej +         *)
				(* wyswietlenie zmierzonych wartosci       *)
	failblocktransmission;	(* wykonanie blednej transmisji blokowej   *)
	interruptedbefore;	(* wykonanie transmisji blokowej przerwanej*)
				(* przez operatora przed startem           *)
	singletransmission;	(* wykonanie pomiaru w trybie pojedynczym  *)
	quit;			(* zakonczenie programu                    *)
end.
